/**
 * \file: mspin_core_callback_adapter.c
 *
 * \version: $Id:$
 *
 * \release: $Name:$
 *
 * MySpin CoreCallbackAdapter
 *
 * \component: MSPIN
 *
 * \author: Torsten Plate ICT-ADITG/SW2 tplate@de.adit-jv.com
 *
 * \copyright: (c) 2003 - 2013 ADIT Corporation
 *
 * \history
 * 0.1 TPlate Initial version
 *
 ***********************************************************************/

#include "mspin_core_callback_adapter.h"
#include "mspin_core_adapter.h"
#include "mspin_lm_adapter.h"
#include "mspin_measurement.h"
#include "mspin_logging.h"

#include <stdio.h>
#include <sys/time.h>

//#define MSPIN_FORCE_FRAME_UPDATE //Enable for testing purpose

extern U32 gLastFrameSize;

static bool gCountFPS = FALSE;
static U32 gFrameUpdates = 0;
static U32 gFrameUpdatesDelta = 0;
static U32 gSavedTimeFrameMeasurement = 0;
// do not set gMillisecsToMeasure to 0
static const U32 gMillisecsToMeasure = 1000;
static U32 gFrameRectUpdates = 0;

void mspin_core_onMainThreadStart(void* context)
{
    mspin_context_t* pMspinContext = (mspin_context_t*)context;

    mspin_measure_reset();

    mspin_log_printLn(eMspinVerbosityDebug,
            "%s(context=%p){%d}",
            __FUNCTION__, pMspinContext, pMspinContext ? pMspinContext->instanceId : (U32)-1);
}

void mspin_core_onMainThreadEnd(void* context)
{
    mspin_context_t* pMspinContext = (mspin_context_t*)context;
    if (pMspinContext && pMspinContext->pLayerManagerContext)
    {
        mspin_log_printLn(eMspinVerbosityDebug,
                "%s(ctx=%p){%d} -> mspin_core_deleteLMContext()",
                __FUNCTION__, context, pMspinContext->instanceId);

        //Inform higher layer not to use LayerManager anymore
        if (pMspinContext->OnLMDeinitializedCB)
        {
            pMspinContext->OnLMDeinitializedCB(pMspinContext->outsideContext);
        }
        mspin_core_deleteLMContext(pMspinContext);

        mspin_log_printLn(eMspinVerbosityDebug,
                "%s(ctx=%p){%d} returns",
                __FUNCTION__, context, pMspinContext->instanceId);
    }
    else if (!pMspinContext)
    {
        mspin_log_printLn(eMspinVerbosityFatal,
                "%s(ctx=%p){} FATAL ERROR: Context is NULL",
                __FUNCTION__, context);
    }
    else
    {
        mspin_log_printLn(eMspinVerbosityWarn,
                "%s(ctx=%p){} WARNING: LM context is NULL",
                __FUNCTION__, context);
    }
}

void mspin_core_onFrameUpdateStart(void *pContext, UInt8 numOfRectangles)
{
    mspin_context_t *pMspinContext = (mspin_context_t*)pContext;
    if (pMspinContext)
    {
        //Reset
        pMspinContext->discardCurrentFrame = FALSE;

        //Discard frame
        if (pMspinContext->appTransitionInProgress)
        {
            mspin_log_printLn(eMspinVerbosityInfo,
                    "%s(ctx=%p, #rects=%d) appTransition in progress -> discard frame",
                    __FUNCTION__, pContext, numOfRectangles);
            pMspinContext->discardCurrentFrame = TRUE;
            return;
        }

        int rc = pthread_mutex_lock(&(pMspinContext->LMCtxMutex));
        if (0 != rc)
        {
            mspin_log_printLn(eMspinVerbosityFatal, "%s() FATAL ERROR: Failed to lock LMCtxMutex with rc=%d",
                    __FUNCTION__, rc);
        }
        if (pMspinContext->pLayerManagerContext && !(pMspinContext->isCanceled))
        {
            //Initialize LayerManager if not yet happened.
            //It should not happen that the old LayerManager is still there. But if it is happening we can't reuse it
            // because the LayerManager was created from another thread and this might cause a crash in eglMakeCurrent()
            // (when using APX)
            if (!pMspinContext->pLayerManagerContext->readyToUse)
            {
                mspin_log_printLn(eMspinVerbosityInfo,
                        "%s(ctx=%p, #rects=%d) frame received, but LayerManager isn't present -> mspin_lm_createContext()",
                        __FUNCTION__, pContext, numOfRectangles);

                mspin_measure_startInitLayer();

                if (MSPIN_SUCCESS != mspin_core_createLMContext(pMspinContext))
                {
                    mspin_log_printLn(eMspinVerbosityError,
                            "%s(ctx=%p, #rects=%d) ERROR: Failed to setup LayerManager",
                            __FUNCTION__, pContext, numOfRectangles);
                    rc = pthread_mutex_unlock(&(pMspinContext->LMCtxMutex));
                    if (0 != rc)
                    {
                        mspin_log_printLn(eMspinVerbosityFatal, "%s() FATAL ERROR: Failed to unlock LMCtxMutex with rc=%d",
                             __FUNCTION__, rc);
                    }
                    return;
                }

                mspin_measure_layerManagerInitialized();

                if (!pMspinContext->pLayerManagerContext->readyToUse)
                {
                    mspin_log_printLn(eMspinVerbosityError,
                            "%s(ctx=%p, #rects=%d) ERROR: Initializing LayerManager failed",
                            __FUNCTION__, pContext, numOfRectangles);
                    rc = pthread_mutex_unlock(&(pMspinContext->LMCtxMutex));
                    if (0 != rc)
                    {
                        mspin_log_printLn(eMspinVerbosityFatal, "%s() FATAL ERROR: Failed to unlock LMCtxMutex with rc=%d",
                             __FUNCTION__, rc);
                    }
                    return;
                }

                mspin_log_printLn(eMspinVerbosityDebug,
                        "%s(ctx=%p, #rects) LayerManager initialized ",
                        __FUNCTION__, pContext, numOfRectangles);

                //Inform higher layer that LayerManager is initialized and ready to use
                if (pMspinContext->OnLMInitializedCB)
                {
                    pMspinContext->OnLMInitializedCB(pMspinContext->outsideContext);
                }
            }
            //else not relevant
        }
        else if (!(pMspinContext->pLayerManagerContext))
        {
            mspin_log_printLn(eMspinVerbosityFatal,
                    "%s(ctx=%p){} FATAL ERROR: LM context is NULL",
                    __FUNCTION__, pContext);
        }
        rc = pthread_mutex_unlock(&(pMspinContext->LMCtxMutex));
        if (0 != rc)
        {
            mspin_log_printLn(eMspinVerbosityFatal, "%s() FATAL ERROR: Failed to lock LMCtxMutex with rc=%d",
                    __FUNCTION__, rc);
        }
    }
    else
    {
        mspin_log_printLn(eMspinVerbosityFatal,
                "%s(ctx=%p){} FATAL ERROR: Context is NULL",
                __FUNCTION__, pContext);
    }

    mspin_measure_frameReceived();
    ++gFrameUpdates;
    ++gFrameUpdatesDelta;

    if (gCountFPS)
    {
       U32 timeMs = 0;
       struct timeval tp;
       (void)gettimeofday (&tp, NULL);
       timeMs = (U32)(tp.tv_sec * 1000) + (U32)(tp.tv_usec / 1000);

       // calc FPS only after gMillisecsToMeasure
       if ( (timeMs - gSavedTimeFrameMeasurement) >= gMillisecsToMeasure)
       {
          U32 tenthFps = gFrameUpdatesDelta * 1000 * 10 / (timeMs - gSavedTimeFrameMeasurement);
          mspin_log_printLn(eMspinVerbosityWarn, " frames per second: %d.%d  with size=%d",
                tenthFps / 10, tenthFps % 10, gLastFrameSize); //use higher log level for fps
          gSavedTimeFrameMeasurement = timeMs;
          gFrameUpdatesDelta = 0;
       }
    }

    mspin_log_printLn(eMspinVerbosityDebug, "%s(ctx=%p, #rects=%d) frame (%d) update start with size=%d",
            __FUNCTION__, pContext, numOfRectangles, gFrameUpdates, gLastFrameSize);
}

void mspin_core_onFrameRectUpdate(void *pContext, UInt8 currentNumber, UInt16 left, UInt16 top, UInt16 width,
        UInt16 height, UInt8 *pBuffer, UInt32 bufferSize)
{
    mspin_context_t *pMspinContext = (mspin_context_t*)pContext;

    //Discard frame during appTransition in progress
    if (pMspinContext && pMspinContext->discardCurrentFrame)
    {
        mspin_log_printLn(eMspinVerbosityDebug,
                "%s(ctx=%p, num=%d, size=%d) discard current frame -> do nothing",
                __FUNCTION__, pContext, currentNumber, bufferSize);
        return;
    }

    ++gFrameRectUpdates;

    mspin_measure_frameRectCompressed((U8)currentNumber, (U32)bufferSize);

    mspin_log_printLn(eMspinVerbosityDebug, "%s(ctx=%p, num=%d, size=%d){%d}",
            __FUNCTION__, pContext, currentNumber, bufferSize, gFrameRectUpdates);

    if (pMspinContext && !(pMspinContext->isCanceled))
    {
        int rc = pthread_mutex_lock(&(pMspinContext->LMCtxMutex));
        if (0 != rc)
        {
            mspin_log_printLn(eMspinVerbosityFatal, "%s() FATAL ERROR: Failed to lock LMCtxMutex with rc=%d",
                    __FUNCTION__, rc);
        }
        mspin_lm_copyRect(pMspinContext->pLayerManagerContext, currentNumber, left, top, width, height, pBuffer, bufferSize);
        rc = pthread_mutex_unlock(&(pMspinContext->LMCtxMutex));
        if (0 != rc)
        {
            mspin_log_printLn(eMspinVerbosityFatal, "%s() FATAL ERROR: Failed to unlock LMCtxMutex with rc=%d",
                    __FUNCTION__, rc);
        }
    }
    else if (!pMspinContext)
    {
        mspin_log_printLn(eMspinVerbosityFatal,
                "%s(ctx=%p) FATAL ERROR: Context is NULL",
                __FUNCTION__, pContext);
    }
}

void mspin_core_onFrameUpdateEnd(void *pContext)
{
    mspin_context_t *pMspinContext = (mspin_context_t*)pContext;

    //Discard frame during appTransition in progress
    if (pMspinContext && pMspinContext->discardCurrentFrame)
    {
        mspin_log_printLn(eMspinVerbosityDebug, "%s(ctx=%p) discard current frame but do not request new one",
                __FUNCTION__, pContext);
        return;
    }

    mspin_measure_frameReadyToDisplay();

    mspin_log_printLn(eMspinVerbosityDebug, "%s(ctx=%p) entered", __FUNCTION__, pContext);

    if (pMspinContext)
    {
        int rc = pthread_mutex_lock(&(pMspinContext->LMCtxMutex));
        if (0 != rc)
        {
            mspin_log_printLn(eMspinVerbosityFatal, "%s() FATAL ERROR: Could not acquire lock with rc=%d\n", __FUNCTION__, rc);
            /* PRQA: Lint Message 454: A thread mutex isn't locked in this case. So ignore this error */
            /*lint -save -e454*/
            return;
            /*lint -restore*/
        }

        if (pMspinContext->pLayerManagerContext && !pMspinContext->isCanceled)
        {
            mspin_lm_drawFrame(pMspinContext->pLayerManagerContext);

            mspin_measure_frameRendered();

            //Request new frame
            if (pMspinContext->pLayerManagerContext->readyToUse && !pMspinContext->isSuspended)
            {
#ifdef MSPIN_FORCE_FRAME_UPDATE
                mspin_core_requestFrameUpdate(pMspinContext, FALSE); //force update
#else
                mspin_core_requestFrameUpdate(pMspinContext, TRUE); //incremental update
#endif //MSPIN_FORCE_FRAME_UPDATE
            }
            else if (!pMspinContext->pLayerManagerContext->readyToUse)
            {
                mspin_log_printLn(eMspinVerbosityError, "%s(ctx=%p) ERROR: LayerManager context is NULL -> do not request new frame",
                        __FUNCTION__, pContext);
            }
            else
            {
                mspin_log_printLn(eMspinVerbosityWarn, "%s(ctx=%p) WARNING: mySPIN suspended -> do not request new frame",
                        __FUNCTION__, pContext);
            }
        }
        else if (!(pMspinContext->pLayerManagerContext))
        {
            mspin_log_printLn(eMspinVerbosityFatal,
                    "%s(ctx=%p) FATAL ERROR: LM context is NULL",
                    __FUNCTION__, pContext);
        }

        rc = pthread_mutex_unlock(&(pMspinContext->LMCtxMutex));
        if (0 != rc)
        {
            mspin_log_printLn(eMspinVerbosityFatal, "%s() FATAL ERROR: Could not release lock with rc=%d\n", __FUNCTION__, rc);
            //return;
        }

        if (pMspinContext->pLayerManagerContext && !pMspinContext->isCanceled)
        {
            //Issue first frame rendered callback if necessary
            if (pMspinContext->issueFirstFrameRendered)
            {
                if (pMspinContext->OnFirstFrameRenderedCB)
                {
                    mspin_log_printLn(eMspinVerbosityDebug, "%s(ctx=%p) call OnFirstFrameRenderedCB",
                            __FUNCTION__, pContext);
                    pMspinContext->OnFirstFrameRenderedCB(pMspinContext->outsideContext);
                }

                pMspinContext->issueFirstFrameRendered = FALSE; //reset value after issuing callback
            }

            //Issue frame transmission status change callback
            if (!pMspinContext->isSuspended)
            {
                mspin_core_updateFrameTransmissionStatus(pMspinContext, eFLAG_TRUE);
            }
            else
            {
                mspin_log_printLn(eMspinVerbosityDebug,
                        "%s(ctx=%p) frame received in suspended state -> do not change frame transmission status",
                        __FUNCTION__, pContext);
            }
        }
    }
    else
    {
        mspin_log_printLn(eMspinVerbosityFatal,
                "%s(ctx=%p) FATAL ERROR: Context is NULL",
                __FUNCTION__, pContext);
    }
}

#if (MSPIN_IVI_CORE_VERSION >= 0x01020000UL) //Supported since IVI Core 1.2.x
static const char* mspin_core_translateLauncherState(MSPIN_LAUNCHER_STATE state)
{
    switch (state)
    {
        case (MSPIN_LAUNCHER_STATE_NOT_VISIBLE):
        {
            return "Launcher not visible";
            //break;
        }
        case (MSPIN_LAUNCHER_STATE_VISIBLE_FIRST_SCREEN):
        {
            return "First screen of launcher visible";
            //break;
        }
        case (MSPIN_LAUNCHER_STATE_VISIBLE_OTHER_SCREEN):
        {
            return "Other screen of launcher visible";
            //break;
        }
        default:
        {
            return "Unknown state";
            //break;
        }
    }
}
#endif //(MSPIN_IVI_CORE_VERSION >= 0x01020000UL)

static const char* mspin_core_translateProtocolState(eProtocolState state)
{
    switch (state)
    {
        case ePROTOCOL_STATE_NONE:
        {
            return "NONE";
            //break;
        }
        case ePROTOCOL_STATE_HANDSHAKE:
        {
            return "Handshake";
            //break;
        }
        case ePROTOCOL_STATE_INIT_CLIENT:
        {
            return "Init Client";
            //break;
        }
        case ePROTOCOL_STATE_INIT_SERVER:
        {
            return "Init Server";
            //break;
        }
#if (MSPIN_IVI_CORE_VERSION >= 0x01020000UL) //New since IVI Core 1.2
        case ePROTOCOL_STATE_INIT_DONE:
        {
            return "Init Done";
            //break;
        }
#endif //(MSPIN_IVI_CORE_VERSION >= 0x01020000UL)
        case ePROTOCOL_STATE_RUN:
        {
            return "Run";
            //break;
        }
        case ePROTOCOL_STATE_STOP:
        {
            return "Stop";
            //break;
        }
        default:
        {
            return "ProtocolStateUnknown";
            //break;
        }
    }
}

static const char* mspin_core_translateCoreError(eErrorCode error)
{
    switch (error)
    {
        case eERRORCODE_NONE:
        {
            return "NONE";
            //break;
        }
        case eERRORCODE_CONNECTION_TIMEOUT:
        {
            return "CONNECTION_TIMEOUT";
            //break;
        }
        case eERRORCODE_CONNECTION_EOF_ZERO:
        {
            return "EOF_CONNECTION_ERROR";
            //break;
        }
        case eERRORCODE_PROTOCOL_STOP:
        {
            return "PROTOCOL_STOP";
            //break;
        }
        case eERRORCODE_Z_DATA_ERROR:
        {
            return "Z_DATA_ERROR";
            //break;
        }
        case eERRORCODE_MEM_ALLOC:
        {
            return "MEMORY_FAULT";
            //break;
        }
        case eERRORCODE_FRAME_SIZE_MISMATCH:
        {
            return "FRAME_SIZE_MISMATCH";
            //break;
        }
        case eERRORCODE_INITIALIZATION_TIMEOUT:
        {
            return "INITIALIZATION_TIMEOUT";
            //break;
        }
        case eERRORCODE_RECEIVER_START_FAILED:
        {
            return "RECEIVER_START_FAILED";
            //break;
        }
        case eERRORCODE_MESSAGE_PROCESSING_FAILED:
        {
            return "MESSAGE_PROCESSING_FAILED";
            //break;
        }
#if (MSPIN_IVI_CORE_VERSION >= 0x01010000UL) //New since IVI Core 1.1
        case eERRORCODE_JPEG_ERROR:
        {
            return "JPEG_ERROR";
            //break;
        }
        case eERRORCODE_INITIALIZATION_ABORT:
        {
            return "INITIALIZATION_ABORT";
            //break;
        }
#endif //(MSPIN_IVI_CORE_VERSION >= 0x01010000UL)
        //Add here additional values when enumeration 'eErrorCode' of 'mySPIN-Core.h' get extended/updated
        default:
        {
            return "UNKNOWN";
            //break;
        }
    }
}

void mspin_core_onPhoneCallStatus(void* context, Flag callStarts)
{
    bool isActive = FALSE;
    if (callStarts == eFLAG_TRUE)
    {
        isActive = TRUE;
    }
    mspin_log_printLn(eMspinVerbosityDebug, "%s() with value %s (ctx=%p)",
            __FUNCTION__, (eFLAG_TRUE == callStarts) ? "TRUE" : "FALSE", context);

    mspin_context_t* pContext = (mspin_context_t*)context;
    if (pContext && pContext->OnPhoneCallStatus)
    {
        pContext->OnPhoneCallStatus(pContext->outsideContext, isActive);
    }
}

void mspin_core_onAppTransitionStatus(void* context, Flag transitionStarts)
{
    bool isActive = FALSE;
    if (transitionStarts == eFLAG_TRUE) //=> transition started
    {
        isActive = TRUE;
    }

    mspin_context_t* pContext = (mspin_context_t*)context;
    if (pContext)
    {
        pContext->appTransitionInProgress = isActive; //update flag

        //Reset force update pending flag
        if (isActive)
        {
            mspin_core_updateFrameTransmissionStatus(pContext, eFLAG_FALSE);    //Issue stop frame transmission
        }

        mspin_log_printLn(eMspinVerbosityDebug, "%s(ctx=%p, %s)",
                __FUNCTION__, context, (eFLAG_TRUE == transitionStarts) ? "transition started" : "transition finished");

        //Issue appTransition callback
        if (pContext->OnAppTransitionStatus)
        {
            pContext->OnAppTransitionStatus(pContext->outsideContext, isActive);
        }

        //Request new frame with force flag in case of appTransition(end)
        if (!isActive && !pContext->isSuspended)
        {
            mspin_log_printLn(eMspinVerbosityDebug, "%s(ctx=%p, %s) and finally request new forced frame",
                    __FUNCTION__, context, (eFLAG_TRUE == transitionStarts) ? "transition started" : "transition finished");

            mspin_core_requestFrameUpdate(pContext, FALSE);
        }
        else if (!isActive && pContext->isSuspended)
        {
            mspin_log_printLn(eMspinVerbosityDebug, "%s(ctx=%p, %s) skip requesting new frame because we are still suspended",
                    __FUNCTION__, context, (eFLAG_TRUE == transitionStarts) ? "transition started" : "transition finished");
        }
    }
    else
    {
        mspin_log_printLn(eMspinVerbosityFatal, "%s(ctx=%p, %s) FATAL ERROR: Context is NULL",
                __FUNCTION__, context, (eFLAG_TRUE == transitionStarts) ? "transition started" : "transition finished");
    }
}

void mspin_core_onAppInactive(void* context, Flag appInactive)
{
    bool isInactive = FALSE;
    if (appInactive == eFLAG_TRUE)
    {
        isInactive = TRUE;
    }

    mspin_log_printLn(eMspinVerbosityDebug, "%s(ctx=%p, %s) entered",
            __FUNCTION__, context, (eFLAG_TRUE == appInactive) ? "inactivity begins" : "inactivity ends");

    //There is no call to mspin_core_updateFrameTransmissionStatus here in case of app inactive
    // begins because this state might be also triggered by popups. But frame transmission can
    // continue while popups are shown

    mspin_context_t* pContext = (mspin_context_t*)context;
    if(pContext)
    {
        if (pContext->OnAppInactive)
        {
            pContext->OnAppInactive(pContext->outsideContext, isInactive);
        }

        //Request new forced frame update on app inactive ends in order to make sure that we get a valid frame as soon as possible
        if (!isInactive && !pContext->isSuspended)
        {
            mspin_log_printLn(eMspinVerbosityDebug, "%s(ctx=%p, %s) and finally request new forced frame",
                    __FUNCTION__, context, (eFLAG_TRUE == appInactive) ? "inactivity begins" : "inactivity ends");

            mspin_core_requestFrameUpdate(pContext, FALSE);
        }
        else if (!isInactive && pContext->isSuspended)
        {
            mspin_log_printLn(eMspinVerbosityDebug, "%s(ctx=%p, %s) skip requesting new frame because we are still suspended",
                    __FUNCTION__, context, (eFLAG_TRUE == appInactive) ? "inactivity begins" : "inactivity ends");
        }
    }

}

#if (MSPIN_IVI_CORE_VERSION < 0x01020000UL) //Not supported since IVI Core 1.2
void mspin_core_onBlockStatus(void* context, Flag isBlocked)
{
    bool blocked = FALSE;
    if (isBlocked == eFLAG_TRUE)
    {
        blocked = TRUE;
    }
    mspin_log_printLn(eMspinVerbosityInfo, "%s() with value %s (ctx=%p)",
            __FUNCTION__, (eFLAG_TRUE == isBlocked) ? "TRUE" : "FALSE", context);

    mspin_context_t* pContext = (mspin_context_t*)context;
    if (pContext && pContext->OnBlockStatus)
    {
        pContext->OnBlockStatus(pContext->outsideContext, blocked);
    }
}
#else //Supported since Core 1.2
void mspin_core_onLauncherState(void* context, LauncherState state)
{
    MSPIN_LAUNCHER_STATE oldState = MSPIN_LAUNCHER_STATE_NOT_VISIBLE;
    mspin_context_t* pContext = (mspin_context_t*)context;

    if (pContext)
    {
        if (state > eLAUNCHERSTATE_VisibleOther)
        {
            mspin_log_printLn(eMspinVerbosityError, "%s(context=%p, launcherState=%d) ERROR: Unsupported LauncherState",
                    __FUNCTION__, context, (int)state);
        }
        else
        {
            oldState = pContext->launcherState;
            pContext->launcherState = (MSPIN_LAUNCHER_STATE)state;

            mspin_log_printLn(eMspinVerbosityInfo, "%s(context=%p, launcherState='%s'(%d)) changed from '%s'(%d)",
                    __FUNCTION__, context, mspin_core_translateLauncherState(pContext->launcherState), (int)pContext->launcherState,
                    mspin_core_translateLauncherState(oldState), (int)oldState);

            //Issue launch state change callback
            if (pContext->OnLauncherStateChangeCB)
            {
                if (oldState != pContext->launcherState)
                {
                    pContext->OnLauncherStateChangeCB(pContext->outsideContext, pContext->launcherState);
                }
            }

            //Issue home button usability change callback
            if (pContext->OnHomeButtonUsabilityChangeCB)
            {
                if (pContext->launcherState != oldState)
                {
                    if (MSPIN_LAUNCHER_STATE_VISIBLE_FIRST_SCREEN == oldState)
                    {
                        pContext->OnHomeButtonUsabilityChangeCB(pContext->outsideContext, true);
                    }
                    else if (MSPIN_LAUNCHER_STATE_VISIBLE_FIRST_SCREEN == pContext->launcherState)
                    {
                        pContext->OnHomeButtonUsabilityChangeCB(pContext->outsideContext, false);
                    }
                }
            }
        }
    }
    else
    {
        mspin_log_printLn(eMspinVerbosityError, "%s(context=%p, launcherState='%s'(%d)) ERROR: Context is NULL",
                            __FUNCTION__, context, mspin_core_translateLauncherState((MSPIN_LAUNCHER_STATE)state), (int)state);
    }
}

void mspin_core_onAppListChanged(void* context)
{
    mspin_log_printLn(eMspinVerbosityInfo, "%s()", __FUNCTION__);

    mspin_context_t* pContext = (mspin_context_t*)context;
    if (pContext && pContext->OnAppListChangedCB)
    {
        pContext->OnAppListChangedCB(pContext->outsideContext);
    }
}

void mspin_core_onAppIconResponse(void* context, AppIconResponse* appIcon)
{
    mspin_log_printLn(eMspinVerbosityInfo, "%s() appIcon=%p", __FUNCTION__, appIcon);

    mspin_context_t* pContext = (mspin_context_t*)context;
    if (pContext && pContext->OnAppIconResponseCB && appIcon)
    {
        MSPIN_AppIconResponse_t appIconResponse;
        appIconResponse.IdentifierURL = appIcon->IdentifierURL;
        appIconResponse.UniqueIconID = appIcon->UniqueIconID;
        appIconResponse.uncompressedDataLength = appIcon->uncompressedDataLength;
        appIconResponse.uncompressedData = appIcon->uncompressedData;
        pContext->OnAppIconResponseCB(pContext->outsideContext, &appIconResponse);
    }
}

void mspin_core_onAppStartedResponse(void* context, const char* identifierURL, AppStartStatus appStartStatus)
{
    mspin_log_printLn(eMspinVerbosityInfo, "%s() with values <%s> <%d>",
            __FUNCTION__, identifierURL, appStartStatus);

    mspin_context_t* pContext = (mspin_context_t*)context;
    if (pContext && pContext->OnAppStartedResponseCB)
    {
        MSPIN_APP_START_STATE state = MSPIN_APPSTARTSTATE_UNKNOWN;
        switch (appStartStatus)
        {
        case eAPPSTARTSTATUS_InProgress:
            state = MSPIN_APPSTARTSTATE_INPROGRESS;
            break;
        case eAPPSTARTSTATUS_Success:
            state = MSPIN_APPSTARTSTATE_SUCCESS;
            break;
        case eAPPSTARTSTATUS_Failed:
            state = MSPIN_APPSTARTSTATE_FAILED;
            break;
        case eAPPSTARTSTATUS_NotInstalled:
            state = MSPIN_APPSTARTSTATE_NOTINSTALLED;
            break;
        case eAPPSTARTSTATUS_NotAllowed:
            state = MSPIN_APPSTARTSTATE_NOTALLOWED;
            break;
        case eAPPSTARTSTATUS_NotPaired:
            state = MSPIN_APPSTARTSTATE_NOTPAIRED;
            break;
        default:
            state = MSPIN_APPSTARTSTATE_UNKNOWN;
            mspin_log_printLn(eMspinVerbosityError, "%s(context=%p){%d} ERROR: illegal appStartStatus %d",
                    __FUNCTION__, pContext, pContext->instanceId, appStartStatus);
            break;
        }
        pContext->OnAppStartedResponseCB(pContext->outsideContext, identifierURL, state);
    }
}
#endif //(MSPIN_IVI_CORE_VERSION < 0x01020000UL)

#if (MSPIN_IVI_CORE_VERSION > 0x01020300UL) //Supported since IVI Core 1.2.3
void mspin_core_onPTTAvailable(void* context, Flag pttAvailable)
{
    mspin_context_t* pContext = (mspin_context_t*)context;
    if (pContext && pContext->OnPTTAvailable)
    {
        mspin_log_printLn(eMspinVerbosityInfo, "%s(ctx=%p, PTT %s) -> issue callback",
                __FUNCTION__, context, (eFLAG_TRUE == pttAvailable) ? "available" : "not available");

        if (eFLAG_TRUE == pttAvailable)
        {
            pContext->OnPTTAvailable(pContext->outsideContext, TRUE);
        }
        else
        {
            pContext->OnPTTAvailable(pContext->outsideContext, FALSE);
        }
    }
    else if (pContext && !pContext->OnPTTAvailable)
    {
        mspin_log_printLn(eMspinVerbosityDebug, "%s(ctx=%p, PTT %s) no callback registered -> ignore",
                __FUNCTION__, context, (eFLAG_TRUE == pttAvailable) ? "available" : "not available");
    }
}
#endif //(MSPIN_IVI_CORE_VERSION > 0x01020300UL)

void mspin_core_onInitiatePhoneCall(void* context, char* numberString, char* displayString)
{
    mspin_log_printLn(eMspinVerbosityInfo, "%s() with values <%s> <%s>",
            __FUNCTION__, numberString, displayString);

    mspin_context_t* pContext = (mspin_context_t*)context;
    if (pContext && pContext->OnInitiatePhoneCall)
    {
        pContext->OnInitiatePhoneCall(pContext->outsideContext, numberString, displayString);
    }
}

#if (MSPIN_IVI_CORE_VERSION >= 0x01010000UL)
void mspin_core_onNavigateTo(void* context, char* displayString, NavigateToType type,
        double longitude, double latitude, LocationStrings* locationDesciption)
{
    mspin_log_printLn(eMspinVerbosityInfo, "%s() with values <%s> <%f> <%f>",
            __FUNCTION__, displayString, longitude, latitude);

    mspin_context_t* pContext = (mspin_context_t*)context;
    if (pContext && pContext->OnNavigateTo)
    {
        MSPIN_NAVIGATETOTYPE validtype = MSPIN_NAVIGATETOTYPE_Both;
        switch (type)
        {
            case eNAVIGATETOTYPE_PositionOnly:
                validtype = MSPIN_NAVIGATETOTYPE_PositionOnly;
                break;
            case eNAVIGATETOTYPE_PlacemarkOnly:
                validtype = MSPIN_NAVIGATETOTYPE_PlacemarkOnly;
                break;
            case eNAVIGATETOTYPE_Both:
                validtype = MSPIN_NAVIGATETOTYPE_Both;
                break;
            default:
                mspin_log_printLn(eMspinVerbosityError,
                        "%s(context=%p){%d} ERROR: illegal NavigateToType %d",
                        __FUNCTION__, pContext, pContext->instanceId, type);
                break;
        }

        MSPIN_LocationStrings_t* pLocation = 0;
        MSPIN_LocationStrings_t location;
        if ((validtype != MSPIN_NAVIGATETOTYPE_PositionOnly) && locationDesciption)
        {
            location.validMask = locationDesciption->validMask;
            location.Country = locationDesciption->Country;
            location.StateRegion = locationDesciption->StateRegion;
            location.ZipCode = locationDesciption->ZipCode;
            location.City = locationDesciption->City;
            location.Street = locationDesciption->Street;
            location.CrossStreet = locationDesciption->CrossStreet;
            location.Housenumber = locationDesciption->Housenumber;
            pLocation = &location;
        }

        pContext->OnNavigateTo(pContext->outsideContext, displayString, validtype, longitude, latitude, pLocation);
    }
}

void mspin_core_onVehicleDataRequest(void* context, Flag request, UInt8 length, UInt32* keyList)
{
    mspin_log_printLn(eMspinVerbosityInfo, "%s() Flag: %d Len: %d",
            __FUNCTION__, request, length);

    mspin_context_t* pContext = (mspin_context_t*)context;
    if (pContext && pContext->OnVehicleDataRequest)
    {
        pContext->OnVehicleDataRequest(pContext->outsideContext, (bool)request, (U8)length, (U32*)keyList);
    }
}

void mspin_core_onCustomString(void* context, ServerCustomDataStringType type, StringEncoding encoding, char* data)
{
    mspin_log_printLn(eMspinVerbosityInfo, "%s(type=%d, data=%s)", __FUNCTION__, type, data);

    mspin_context_t* pContext = (mspin_context_t*)context;
    if (pContext && pContext->OnCustomString)
    {
        pContext->OnCustomString(pContext->outsideContext, (MSPIN_PHONEMSGSTRTYPE)type, (MSPIN_STRINGENCODING)encoding, data);
    }
}

void mspin_core_onVoiceSessionRequest(void* context, VoiceRequest requestType)
{
    mspin_context_t* pContext = (mspin_context_t*)context;
    if (pContext && pContext->OnVoiceSessionRequest)
    {
        if ((eVOICEREQUEST_StartMicAndSpeakers == requestType) || (eVOICEREQUEST_StartMicOnly == requestType))
        {
            mspin_log_printLn(eMspinVerbosityInfo, "%s(context=%p, type=%d) start voice session", __FUNCTION__, context, requestType);
        }
        else if (eVOICEREQUEST_Active == requestType)
        {
            mspin_log_printLn(eMspinVerbosityInfo, "%s(context=%p, type=%d) voice session active", __FUNCTION__, context, requestType);
        }
        else if ((eVOICEREQUEST_EndUserEvent == requestType) || (eVOICEREQUEST_EndRealCallEvent == requestType))
        {
            mspin_log_printLn(eMspinVerbosityInfo, "%s(context=%p, type=%d) end voice session", __FUNCTION__, context, requestType);
        }
        else
        {
            mspin_log_printLn(eMspinVerbosityError, "%s(context=%p, type=%d) ERROR: Unknown request type", __FUNCTION__, context, requestType);
            return;
        }

        pContext->OnVoiceSessionRequest(pContext->outsideContext, (MSPIN_VOICESESSION_REQUESTTYPE)requestType);
    }
    else
    {
        mspin_log_printLn(eMspinVerbosityError, "%s() ERROR: Callback not registered", __FUNCTION__);
    }
}

void mspin_core_onAudioRequest(void* context, AudioControl command, UInt32 requestID, AudioType type)
{
    mspin_context_t* pContext = (mspin_context_t*)context;
    if (pContext && pContext->OnAudioRequest)
    {
        mspin_log_printLn(eMspinVerbosityInfo, "%s(context=%p, command=%d, id=%d, type=%d) audio %s",
                __FUNCTION__, context, command, requestID, type, (command==eAUDIOCONTROL_Request) ? "request" : "release");
        pContext->OnAudioRequest(pContext->outsideContext, (MSPIN_AUDIOCONTROL)command, requestID, (MSPIN_AUDIOTYPE)type);
    }
    else
    {
        mspin_log_printLn(eMspinVerbosityError, "%s() ERROR: Callback not registered", __FUNCTION__);
    }
}

#else
void mspin_core_onCustomString(void* context, ServerCustomMessageType type, char* data)
{
    mspin_log_printLn(eMspinVerbosityInfo, "%s(type=%d, data=%s)", __FUNCTION__, type, data);

    mspin_context_t* pContext = (mspin_context_t*)context;
    if (pContext && pContext->OnCustomString)
    {
        pContext->OnCustomString(pContext->outsideContext, (MSPIN_PHONEMSGSTRTYPE)type,
                MSPIN_STRINGENCODING_UTF8, data); //Core 1.0.x used always UTF8 encoding
    }
}
#endif // (MSPIN_IVI_CORE_VERSION >= 0x01010000UL)

void mspin_core_onFrameUpdateRaw(void* context, UInt8 rectNumber, UInt16 x0, UInt16 y0, UInt16 width, UInt16 height,
        UInt8* buffer, UInt32 bufferSize, UInt8 scaleHorizontal, UInt8 scaleVertical,
        PixelFormat format, PixelEndianess endian, FrameCompression compression)
{
    mspin_context_t* pContext = (mspin_context_t*)context;

    mspin_log_printLn(eMspinVerbosityDebug, "%s", __FUNCTION__);

    //Discard frame during appTransition in progress
    if (pContext && pContext->discardCurrentFrame)
    {
        mspin_log_printLn(eMspinVerbosityDebug,
                "%s(ctx=%p, rectNumber=%u) discard current frame set -> do nothing",
                __FUNCTION__, context, rectNumber);
        return;
    }

    ++gFrameRectUpdates;

    if (pContext && pContext->OnFrameUpdateRawCB)
    {
        pContext->OnFrameUpdateRawCB(pContext->outsideContext, rectNumber, x0, y0, width, height,
                buffer, bufferSize, scaleHorizontal, scaleVertical,
                (MSPIN_PIXEL_FORMAT)format, endian, (MSPIN_FRAME_COMPRESSION)compression);
    }
}

void mspin_core_onCustomInt(void* context, ServerCustomDataIntType type, UInt16 length, UInt8* data)
{
    mspin_log_printLn(eMspinVerbosityInfo, "%s() with values <%d> <%d>",
            __FUNCTION__, type, length);

    mspin_context_t* pContext = (mspin_context_t*)context;
    if (pContext && pContext->OnCustomInt)
    {
        pContext->OnCustomInt(pContext->outsideContext, (MSPIN_PHONEMSGINTTYPE)type, length, data);
    }
}

void mspin_core_onError(void* context, eErrorCode error)
{
    mspin_context_t* pContext = (mspin_context_t*)context;

    mspin_log_printLn(eMspinVerbosityError, "%s(ctx=%p, error=%d){%d} ERROR: IVI Core reports error '%s'",
            __FUNCTION__, context, error, pContext ? pContext->instanceId : (U32)-1, mspin_core_translateCoreError(error));

    if (pContext)
    {
        MSPIN_ERROR err = MSPIN_ERROR_UNKNOWN;
        switch (error)
        {
            case eERRORCODE_NONE:
                err = MSPIN_SUCCESS;
                break;
            case eERRORCODE_CONNECTION_EOF_ZERO:
                err = MSPIN_ERROR_CONNECTION_EOF_ZERO;
                break;
            case eERRORCODE_CONNECTION_TIMEOUT:
                err = MSPIN_ERROR_CONNECTION_TIMEOUT;
                break;
            case eERRORCODE_MEM_ALLOC:
                err = MSPIN_ERROR_MEM_ALLOC;
                break;
            case eERRORCODE_PROTOCOL_STOP:
                err = MSPIN_ERROR_PROTOCOL_STOP;
                break;
            case eERRORCODE_Z_DATA_ERROR:
                err = MSPIN_ERROR_Z_DATA_ERROR;
                break;
            case eERRORCODE_FRAME_SIZE_MISMATCH:
                err = MSPIN_ERROR_FRAME_SIZE_MISMATCH;
                break;
            case eERRORCODE_INITIALIZATION_TIMEOUT:
                err = MSPIN_ERROR_PROTOCOL_SETUP;
                break;
            case eERRORCODE_RECEIVER_START_FAILED:
                err = MSPIN_ERROR_RECEIVER_START_FAILED;
                break;
            case eERRORCODE_MESSAGE_PROCESSING_FAILED:
                err = MSPIN_ERROR_MESSAGE_PROCESSING_FAILED;
                break;
            case eERRORCODE_JPEG_ERROR:
                err = MSPIN_ERROR_JPEG;
                break;
            case eERRORCODE_INITIALIZATION_ABORT:
                err = MSPIN_ERROR_INITIALIZATION_ABORT;
                break;
            default:
                err = MSPIN_ERROR_UNKNOWN;
                break;
        }

        if (MSPIN_SUCCESS != err)
        {
            pContext->isCanceled = TRUE;
            pContext->cancelReason = err;
            if (pContext->pConnectionHandle)
            {
                pContext->pConnectionHandle->connectionError = TRUE;
            }
        }

        (void)sem_post(&(pContext->coreStartingLock));  // wake up a possible waiting thread

        if (pContext->OnError)
        {
            pContext->OnError(pContext->outsideContext, err);
        }
    }
}

void mspin_core_onProtocol(void* context, eProtocolState newState)
{
    mspin_context_t* pContext = (mspin_context_t*)context;

    mspin_log_printLn(eMspinVerbosityDebug, "%s(ctx=%p, state=%s(%d)){%d}",
            __FUNCTION__, context, mspin_core_translateProtocolState(newState), newState,
            pContext ? pContext->instanceId : (U32)-1);

    if (pContext)
    {
        // The protocol status is ignored until the first time _RUN is reached.
        // Same procedure if the cores into STOP state - reverse the Ready flag.
        // This "Ready" flag indicates that the core is ready for FrameUpdate requests.
        switch (newState)
        {
            case ePROTOCOL_STATE_RUN:
            {
                if (FALSE == pContext->isReady)
                {
                    pContext->isCanceled = FALSE; //reset
                    pContext->isReady = TRUE;
                    pContext->hasEverBeenReady = TRUE;
                    sem_post(&(pContext->coreStartingLock));
                }
                else
                {
                    mspin_log_printLn(eMspinVerbosityError,
                            "%s(context=%p, state=%s(%d)){%d} ERROR: Core is ready! Wrong flag",
                            __FUNCTION__, context, mspin_core_translateProtocolState(newState),
                            newState, pContext->instanceId);
                }
                break;
            }
            case ePROTOCOL_STATE_STOP:
            {
                // One of the very first calls to this callback will have status STOP.
                // We need to avoid "false positive" ready here.
                if (pContext->hasEverBeenReady == TRUE)
                {
                    if (pContext->isReady == TRUE)
                    {
                        pContext->isReady = FALSE;
                    }
                }
                break;
            }
            default:
            {
                // do nothing.
                break;
            }
        }
    }
}

void mspin_core_onTraceOutput(void* context, eTraceClass traceClass, char* traceString)
{
    tMspinVerbosityLevel level = eMspinVerbosityOff;
    MSPIN_UNUSED(context);

    switch (traceClass)
    {
    case eTRACECLASS_ERROR:
        level = eMspinVerbosityError;
        break;
    case eTRACECLASS_WARNING:
        level = eMspinVerbosityWarn;
        break;
    case eTRACECLASS_INFORMATION:
        level = eMspinVerbosityInfo;
        break;
    case eTRACECLASS_VERBOSE:
        level = eMspinVerbosityDebug;  // we map VERBOSE to Debug (since Core has no "debug" class)
        break;
    case eTRACECLASS_INF_FRAME: // fall through
    case eTRACECLASS_VER_FRAME:
        level = eMspinVerbosityVerbose;
        break;
    default:
        level = eMspinVerbosityInfo;
        break;
    }
    mspin_log_printPrefixedLine(mspin_log_getCoreDLTContext(), level, "CORE: ", "%s%s",
            mySpin_getTraceClassPrefix(traceClass), traceString);
}

void mspin_core_countFPS(void)
{
    gCountFPS = TRUE;
}

void mspin_core_updateFrameTransmissionStatus(void* context, Flag start)
{
    mspin_context_t* pContext = (mspin_context_t*)context;
    if (pContext)
    {
        if (start != pContext->frameTransmissionActive)
        {
            mspin_log_printLn(eMspinVerbosityDebug, "%s(ctx=%p, %s) state changes",
                    __FUNCTION__, context, (TRUE == start) ? "start" : "stop");

            pContext->frameTransmissionActive = start;
            if (pContext->OnFrameTransmissionStatusChangeCB)
            {
                pContext->OnFrameTransmissionStatusChangeCB(pContext->outsideContext, start);
            }
        }
        else
        {
            mspin_log_printLn(eMspinVerbosityVerbose, "%s(ctx=%p, %s) no update",
                    __FUNCTION__, context, (TRUE == start) ? "start" : "stop");
        }
    }
    else
    {
        mspin_log_printLn(eMspinVerbosityFatal, "%s(ctx=%p, %s) FATAL ERROR: Context is NULL",
                __FUNCTION__, context, (TRUE == start) ? "start" : "stop");
    }
}
